package tivonage;

import java.text.DecimalFormat;
import com.tivo.hme.bananas.BText;
import com.tivo.hme.bananas.BView;
import com.tivo.hme.sdk.HmeEvent;
import com.tivo.hme.sdk.StreamResource;
import com.tivo.hme.sdk.View;

/**
 * Provides the black bar at the bottom of the screen for displaying playback progress
 * 
 * @author <a href="http://www.martiansoftware.com/contact.html">Marty Lamb</a>
 */
public class PlayBar extends com.tivo.hme.bananas.BView {

	// this is just awful.  it's the result of a lot of trial and error, and
	// the end result looks OK, but there are at least a hundred better ways
	// to go about this
	public static final java.awt.Color PROGRESSCOLOR_DEFAULT = new java.awt.Color(0, 192, 0);
	private static final int PREFERRED_W = 546;
	private static final int PREFERRED_H = 52;
	private static final int PLAYBAR_H = 28;
	private static final int PREFERRED_X = 10 + (640 - PREFERRED_W) / 2;

	private static final int PREFERRED_Y = 390;
	private static final int PLAYBAR_Y_OFFSET = 12;
		
	private static final int MARGIN = 30;
	//private static final int PLAYBAR_TOP = SCREEN_HEIGHT - SAFE_TITLE_V - MARGIN - PLAYBAR_HEIGHT;
	private static final int TIMEWIDTH = 50;
	private static final int SHUTTLEPART_HEIGHT = 15;
	
	private static final int SHUTTLE_WIDTH = (int) (TIMEWIDTH * 1.5);
	private static final int SHUTTLE_HEIGHT = PREFERRED_H;
	
	private BView playbar;
	private BView playbarBorder;
	private BView playbarProgress;
	private BView playbarShuttle;
	private BText playbarStart;
	private BText playbarEnd;

	private BView playbarShuttleTop;
	private BView playbarShuttleBottom;
	private BText playbarShuttleTime;
	private BView playbarShuttleIcon;
	
	private View streamView;
	
	private int duration;
	private int progress;
	private int position;
	
	private String streamURL;
	private StreamResource stream;
	private boolean paused;
	
	private DecimalFormat timeFormat;
	private boolean ignorePositionUpdates = false;

	/**
	 * Creates a new PlayBar in the specified parent (probably "normal")
	 * @param parent
	 */
	public PlayBar(BView parent) {
		super(parent, PREFERRED_X, PREFERRED_Y, PREFERRED_W, PREFERRED_H);
		init();
	}
	
	/**
	 * Sets the (normally green) buffer bar color.  PROGRESSCOLOR_DEFAULT is the
	 * default color.
	 * 
	 * @param color the color to paint the buffer bar
	 */
	public void setBarColor(java.awt.Color color) {
		playbarProgress.setResource(color);
		playbarProgress.flush();
	}

	public boolean handleEvent(HmeEvent event) {
		if (!ignorePositionUpdates && (stream != null) && (event.id == stream.getID()) && (event instanceof HmeEvent.ResourceInfo)) {
			HmeEvent.ResourceInfo revt = (HmeEvent.ResourceInfo) event;
			if (revt.status == HmeEvent.ResourceInfo.RSRC_STATUS_COMPLETE) {
				ignorePositionUpdates = true;
				setPosition(duration);
				return (true);
			}
			try {
				String[] s = ((String) revt.map.get("pos")).split("/");
				setPosition(Integer.parseInt(s[0]) / 1000);
			} catch (Exception e) {
				// ignore exceptions.  this is very lazy code.
				// e.printStackTrace();
			}
			return (super.handleEvent(event));
		}
		return (super.handleEvent(event));
	}

	/**
	 * Stops playback
	 */
	public void stop() {
		if (stream != null) {
			try {stream.setSpeed(0.0f);} catch (Throwable t) {}
			stream = null;
		}
	}

	/**
	 * Pauses or unpauses the stream
	 * @param paused if true, will pause the stream
	 */
	public void setPaused(boolean paused) {
		if (paused) {
			try {
				stream.setSpeed(0.0f);
				stream.flush();
			} catch (Exception ignore) {}
			playbarShuttleIcon.setResource("pause.png");
		} else {
			try {
				stream.setSpeed(1.0f);
				stream.flush();
			} catch (Exception ignore) {}
			playbarShuttleIcon.setResource("play.png");
		}
		playbarShuttleIcon.flush();
		this.paused = paused;
	}
	
	public boolean handleKeyPress(int code, long rawcode) {
		
		switch(code) {
			case KEY_PAUSE:
				if (paused) {
					ignorePositionUpdates = false;
				}
				setPaused(!paused);
				return (true);
			case KEY_PLAY:
				if (paused) {
					setPaused(false);
					ignorePositionUpdates = false;
				}
				return (true);
			case KEY_ADVANCE:
				play("select.snd");
				if (position == duration) {
					boolean wasPaused = paused;
					play();
					if (wasPaused) {
						setPaused(true);
					}
				} else {
					stream.setSpeed(0f);
					ignorePositionUpdates = true;
					setPosition(duration); 
				}
				return (true);
            case KEY_FORWARD:
            case KEY_REVERSE:
			case KEY_REPLAY:
			case KEY_SLOW:
			case KEY_RECORD:
						play("bonk.snd");
            			return (true);
		}
		return (super.handleKeyPress(code, rawcode));
		
	}

	private void init() {
		timeFormat = new DecimalFormat("00");
		playbar = new BView(this, 0, PLAYBAR_Y_OFFSET, this.width, PLAYBAR_H, false);
		playbar.setResource("playbar.png");
//		playbar.setResource(java.awt.Color.RED);

		streamView = new View(playbar, 0, 0, 1, 1);
		
		playbarStart = new BText(playbar, 0, 0, TIMEWIDTH, 25);
		playbarStart.setFont("default-15.font");
		playbarStart.setFlags(RSRC_HALIGN_CENTER | RSRC_VALIGN_BOTTOM);
		playbarStart.setValue(formatTime(0));

		playbarEnd = new BText(playbar, this.width - TIMEWIDTH, 0, TIMEWIDTH, 25);
		playbarEnd.setFont("default-15.font");
		playbarEnd.setFlags(RSRC_HALIGN_CENTER | RSRC_VALIGN_BOTTOM);
		playbarEnd.setValue(formatTime(0));
		
		playbarProgress = new BView(playbar, TIMEWIDTH, 8, this.width - 2 * TIMEWIDTH + 2, PLAYBAR_H - 16);
		playbarProgress.setResource(PROGRESSCOLOR_DEFAULT);
		setProgress(0);
		
		int[] tickX = { 112, 223, 334};
		for (int i = 0; i < 3; ++i) {
			BView playbarTick = new BView(playbar, TIMEWIDTH + tickX[i], 8, 2, 3);
			playbarTick.setResource(java.awt.Color.WHITE);
		}
		playbar.setVisible(true);

		playbarShuttle = new BView(this, TIMEWIDTH - (SHUTTLE_WIDTH/2), 0, SHUTTLE_WIDTH, SHUTTLE_HEIGHT);
		
		playbarShuttleTop = new BView(playbarShuttle, 0, 0, SHUTTLE_WIDTH, SHUTTLEPART_HEIGHT);
		playbarShuttleTop.setResource("playbar_shuttle_top.png");

		
		playbarShuttleTime = new BText(playbarShuttle, 0, 0, SHUTTLE_WIDTH, 18);
		playbarShuttleTime.setFont("default-15.font");
		playbarShuttleTime.setFlags(RSRC_HALIGN_CENTER | RSRC_VALIGN_CENTER);
		playbarShuttleTime.setValue(formatTime(0));
		
		playbarShuttleBottom = new BView(playbarShuttle, 0, SHUTTLE_HEIGHT - SHUTTLEPART_HEIGHT, SHUTTLE_WIDTH, SHUTTLEPART_HEIGHT);
		playbarShuttleBottom.setResource("playbar_shuttle_bottom.png");
		
		playbarShuttleIcon = new BView(playbarShuttleBottom, (SHUTTLE_WIDTH - 15) / 2, 2, 15, 15);
		playbarShuttleIcon.setResource("pause.png");
		
		BView playbarShuttleNeedle = new BView(playbarShuttle, SHUTTLE_WIDTH / 2, SHUTTLEPART_HEIGHT + 5, 2, PLAYBAR_H - 16);
		playbarShuttleNeedle.setResource(java.awt.Color.WHITE);
		
		setFocusable(true);
	}

	/**
	 * Sets the stream to play
	 * @param streamURL the stream to play
	 */
	public void setStream(String streamURL) {
		this.streamURL = streamURL;
	}

	private String formatTime(int secs) {
		if (duration > 3600) secs = secs / 60;
		return ((secs / 60) + ":" + timeFormat.format(secs % 60));
	}

	/**
	 * Plays the stream set by setStream()
	 */
	public void play() {
		setPosition(0);
		ignorePositionUpdates = false;

		paused = false;
		stream = createStream(streamURL);

		streamView.setResource(stream);
		setPaused(false);
	}
	
	/**
	 * Sets the current playback position, in seconds
	 * @param seconds the current playback position
	 */
	public void setPosition(int seconds) {
		if (seconds == position) return;
		
		int centerOffset = 0;
		int maxWidth = this.width - 2 * TIMEWIDTH;
		
		position = Math.max(seconds, 0);
		position = Math.min(position, progress);

		if (duration >= 0) {
			centerOffset = (int) ((((float) position) / ((float) duration)) * maxWidth);  
		}
		playbarShuttleTime.setValue(formatTime(position));
		playbarShuttle.setLocation(TIMEWIDTH - (SHUTTLE_WIDTH/2) + centerOffset, playbarShuttle.y);

		playbarShuttle.flush();		
		playbarShuttleTime.flush();
	}
	
	/**
	 * Sets the buffer "progress", or the number of seconds of data buffered.
	 * @param seconds the amount of data buffered
	 */
	public void setProgress(int seconds) {
		int newWidth = 0;
		int maxWidth = 446;

		progress = Math.max(seconds, 0);
		progress = Math.min(progress, duration);

		if (duration >= 0) {
			newWidth = (int) ((((float) progress) / ((float) duration)) * maxWidth);  
		}
		playbarProgress.setSize(Math.min(newWidth, maxWidth), playbarProgress.height);
		playbarProgress.flush();
	}
	
	public int getProgress() {
		return (progress);
	}

	/**
	 * Sets the duration of the stream to play.  This determines the displayed
	 * endpoints of the playbar
	 * 
	 * @param duration the duration of the stream in seconds
	 */
	public void setDuration(int duration) {
		this.duration = duration;
		playbarEnd.setValue(formatTime(duration));
		setPosition(position);
	}
	
	public int getDuration() {
		return (duration);
	}
}


