package tivonage;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.LineNumberReader;
import java.util.Iterator;
import java.util.Properties;

import javax.swing.JOptionPane;

import com.mutchek.vonaje.Message;
import com.mutchek.vonaje.VonagePhoneNumber;
import com.mutchek.vonaje.VonageAccount;

/**
 * Stores the configuration for TiVonage, prompting the user for
 * input if necessary.
 * 
 * @author <a href="http://www.martiansoftware.com/contact.html">Marty Lamb</a>
 */
class Config {
	private static final String USERNAME_KEY = "user";
	private static final String PASSWORD_KEY = "password";
	private static final String PHONENUMBER_KEY = "phonenum";
	private static final String COUNTRY_KEY = "country";
	private static final String MP3ENCODER_KEY = "encoder";
	private static final String CHECKFORUPDATES_KEY = "checkforupdates";
	
	private static boolean showedWizard = false;
    private File configFile;
    private Properties props;
    
    private static final String DEFAULT_ENCODER = (isWindows() ? "encode.bat" : "encode.sh") + " $1 $2";
    
    private static boolean isWindows() {
    	return(System.getProperty("os.name", "").toLowerCase().contains("windows"));
    }

    /**
     * Creates a new Config by loading the specified file
     * @param configFile the file to load
     */
    public Config(File configFile) {
    	this.configFile = configFile;
    	props = new Properties();
    	if (configFile.exists()) {
    		try {
    			props.load(new java.io.BufferedInputStream(new java.io.FileInputStream(configFile)));

    			if (isConfigured()) {
    				System.setProperty("vonaje.country", props.getProperty(COUNTRY_KEY));
    			}
    		} catch (java.io.IOException e) {
    			handleError("Unable to load settings from " + configFile.getAbsolutePath(), e);
    		}
    		
    	}
    }

    /**
     * Returns the vonage web interface username
     * @return the vonage web interface username
     */
    public String getUsername() {
    	return (props.getProperty(USERNAME_KEY));
    }
    
    /**
     * Returns the vonage web interface password
     * @return the vonage web interface password
     */
    public String getPassword() {
    	return (props.getProperty(PASSWORD_KEY));
    }

    /**
     * Returns the vonage phone number
     * @return the vonage phone number
     */
    public String getPhoneNumber() {
    	return (props.getProperty(PHONENUMBER_KEY));
    }
    
    /**
     * Returns the mp3 encoder command
     * @return the mp3 encoder command
     */
    public String getMp3Encoder() {
    	return (props.getProperty(MP3ENCODER_KEY));
    }

    public String getCountry() {
    	return (props.getProperty(COUNTRY_KEY));
    }
    
    public boolean isUpdateCheckEnabled() {
    	return (props.getProperty(CHECKFORUPDATES_KEY, "N").equalsIgnoreCase("Y"));
    }
    
    /**
     * Returns true iff this Config is fully configured
     * @return true iff this Config is fully configured
     */
    public boolean isConfigured() {
    	return (!isEmpty(getUsername())
    			&& !isEmpty(getPassword())
				&& !isEmpty(getPhoneNumber())
				&& !isEmpty(getMp3Encoder())
				&& !isEmpty(getCountry())
				&& !isEmpty(props.getProperty(CHECKFORUPDATES_KEY)));
    }
    
    /**
     * Returns true if the specified String is empty (i.e., null, zero length, or only whitespace)
     * @param s the string to check
     * @return true if s is empty
     */
    private boolean isEmpty(String s) {
    	return ((s == null) || (s.trim().length() == 0));
    }
    
    /**
     * Prompts the user for input
     * @param in the input source
     * @param msg the prompt
     * @param defaultValue the default value to return if the user just hits enter
     * @return the user's input
     * @throws IOException probably never
     */
    private String textPrompt(LineNumberReader in, String msg, String defaultValue) throws IOException {
    	String emptyMsg = "";
    	if (!isEmpty(defaultValue)) {
    		emptyMsg = " (press Enter to keep \"" + defaultValue + "\") ";
    	}
    	System.out.print(msg + emptyMsg);
    	String result = in.readLine();
    	return (isEmpty(result) ? defaultValue : result);
    }

    private String prompt(LineNumberReader ignored, String msg, String defaultValue) throws IOException {
    	String result = JOptionPane.showInputDialog(null, msg, defaultValue);
//    	String emptyMsg = "";
//    	if (!isEmpty(defaultValue)) {
//    		emptyMsg = " (press Enter to keep \"" + defaultValue + "\") ";
//    	}
//    	System.out.print(msg + emptyMsg);
//    	String result = in.readLine();
    	return (isEmpty(result) ? defaultValue : result);
    }

    /**
     * Runs the user through the configuration wizard
     */
    public void showWizard() {
    	if (showedWizard) return;
    	try {
	    	LineNumberReader in = new LineNumberReader(new java.io.InputStreamReader(System.in));
	    	for (int i = 0; i < 25; ++i) System.out.println();
	    	System.out.println("TiVonage Configuration");
	    	System.out.println("(stored in " + configFile.getAbsolutePath() + ")\n");
	    	
	    	boolean vonageOK = false;
	    	
	    	String s;
	    	
	    	while (!vonageOK) {
		    	do {
		    		s = prompt(in, "\nStep 1 of 7 - Vonage username: ", props.getProperty(USERNAME_KEY));
		    	} while (isEmpty(s));
		    	props.setProperty(USERNAME_KEY, s);
		    	
		    	do {
		    		s = prompt(in, "\nStep 2 of 7 - Vonage password: ", props.getProperty(PASSWORD_KEY));
		    	} while (isEmpty(s));
		    	props.setProperty(PASSWORD_KEY, s);
		    	
		    	do {
		    		s = prompt(in, "\nStep 3 of 7 - Vonage phone number (format: 12125551212): ", props.getProperty(PHONENUMBER_KEY));
		    	} while (isEmpty(s));
		    	props.setProperty(PHONENUMBER_KEY, s);
		    	
		    	do {
		    		s = prompt(in, "\nStep 4 of 7 - Vonage Country Code (\"us\" for United States, \"ca\" for Canada)", props.getProperty(COUNTRY_KEY, java.util.Locale.getDefault().getCountry().toLowerCase()));
		    	} while (isEmpty(s));
		    	props.setProperty(COUNTRY_KEY, s);
		    	System.setProperty("vonaje.country", s);
		    	
		    	System.out.print("\nStep 5 of 7 - Testing settings...");
		    	try {
		    		VonagePhoneNumber number = new VonagePhoneNumber(new VonageAccount(props.getProperty(USERNAME_KEY), props.getProperty(PASSWORD_KEY)), props.getProperty(PHONENUMBER_KEY));
		    		java.util.List messages = number.getMessageList();
		    		System.out.println(messages.size() + " message(s).");
		    		for (Iterator i = messages.iterator(); i.hasNext();) {
		    			Message msg = (Message) i.next();
		    			System.out.println("   " + msg.getDateString() + "     " + msg.getCallerNumber()); 
		    		}
		    		System.out.println("Vonage settings OK.");
					vonageOK = true;
		    	} catch (Exception e) {
		    		System.out.println(" uh-oh.");
		    		System.err.println("\nUnable to retrieve information from Vonage.");
		    		System.err.println("Stack trace follows:");
		    		e.printStackTrace();
		    	}
		    	
	    	}
	    	
	    	
	    	do {
	    		String msg = "Step 6 of 7 - MP3 encoder command line:\n"
	    					+ "(sorry, this version does NOT support spaces in the command.\n"
							+ "Use $1 as the source .WAV file parameter and $2 as the destination MP3 file parameter.)\n"
							+ "Example (Windows): encode.bat $1 $2\n"
							+ "Example (Non-Windows): encode.sh $1 $2\n";
	    		s = prompt(in, msg + "MP3 encoder command line: ", props.getProperty(MP3ENCODER_KEY, DEFAULT_ENCODER));
	    	} while (isEmpty(s));
	    	props.setProperty(MP3ENCODER_KEY, s);
	    	
	    	String title = "Step 7 of 7 - Check for Updates";
	    	String msg = "Should TiVonage automatically check for updates?\nUpdate checks are performed anonymously.";
	    	if (JOptionPane.YES_OPTION == JOptionPane.showConfirmDialog(null, msg, title, JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE)) {
	    		props.setProperty(CHECKFORUPDATES_KEY, "Y");
	    	} else {
	    		props.setProperty(CHECKFORUPDATES_KEY, "N");
	    	}
	
    		FileOutputStream f = new FileOutputStream(configFile);
    		props.store(f, null);
    		f.close();
    	} catch (java.io.IOException e) {
    		handleError("Unable to save settings to " + configFile.getAbsolutePath(), e);
    	}
    	showedWizard = true;
    }

    private void handleError(String msg, Throwable t) {
    	System.err.println("***");
    	System.err.println(msg);
    	System.err.println("***");
    	if (t != null) t.printStackTrace();
    }

}
