package martianutils.lang;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Timer;
import java.util.TimerTask;

/**
 * A helper class for launching external processes and capturing their results.
 * 
 * GOTCHA: If a program has a LOT of stderr output, there might be problems, as
 * stderr isn't read until stdout is closed.  This is probably fine in most cases
 * as a little bit of stderr will be bufferred.  If more is necessary, the waitFor
 * method will have to be reimplemented to use nio channels.
 *  
 * @author <a href="http://www.martiansoftware.com/contact.html">Marty Lamb</a>
 */
public class ProcessResult {

	private static final int BUFSIZE = 256;
	private InputStream out;
	private InputStream err;
	private byte[] outBytes;
	private byte[] errBytes;
	private Process process;
	private long maxRuntimeMillis = 0;
	private boolean killed = false;
	private static Timer timer = new Timer(true);
	
	/**
	 * Creates a new ProcessResult by executing the specified process.
	 * @param process the process to execute
	 * @param maxRuntimeMillis if &gt; 0, the maximum runtime to allow the process, in milliseconds
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public ProcessResult(Process process, long maxRuntimeMillis) throws IOException {
		init(process, maxRuntimeMillis);
	}
	
	/**
	 * Creates a new ProcessResult by executing the specified process.
	 * @param process the process to execute
	 * @throws IOException
	 * @throws InterruptedException
	 */
//	public ProcessResult(Process process) throws IOException {
//		init(process, 0);
//	}

	/**
	 * Creates a new ProcessResult by executing the specified command.  This is just
	 * a convenience method that calls Runtime.getRuntime().exec(command)
	 * @param command the command to execute
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public ProcessResult(String command) throws IOException {
		init(Runtime.getRuntime().exec(command), 0);
	}

	/**
	 * Creates a new ProcessResult by executing the specified command.  This is just
	 * a convenience method that calls Runtime.getRuntime().exec(command)
	 * @param command the command to execute
	 * @param maxRuntimeMillis if &gt; 0, the maximum runtime to allow the process, in milliseconds
	 * @throws IOException
	 * @throws InterruptedException
	 */
	public ProcessResult(String command, long maxRuntimeMillis) throws IOException {
		init(Runtime.getRuntime().exec(command), maxRuntimeMillis);
	}
	
	private void init(Process process, long maxRuntimeMillis) {
		this.process = process;
		this.maxRuntimeMillis = maxRuntimeMillis;
		out = process.getInputStream();
		err = process.getErrorStream();
		if (maxRuntimeMillis > 0) {
			timer.schedule(new ProcessKiller(this), maxRuntimeMillis);
		}
//		waitFor();
	}

	/**
	 * Returns the process that was executed
	 * @return the process that was executed
	 */
	public Process getProcess() {
		return (process);
	}
	
	/**
	 * Returns the stdout output of the process as an array of bytes
	 * @return the stdout output of the process as an array of bytes
	 */
	public byte[] getOutputBytes() {
		return (outBytes);
	}
	
	/**
	 * Returns the stdout output of the process as a String
	 * @return the stdout output of the process as a String
	 */
	public String getOutputString() {
		return (new String(getOutputBytes()));
	}
	
	/**
	 * Returns the stderr output of the process as an array of bytes
	 * @return the stderr output of the process as an array of bytes
	 */
	public byte[] getErrorBytes() {
		return (errBytes);
	}

	/**
	 * Returns the stderr output of the process as a String
	 * @return the stderr output of the process as a String
	 */
	public String getErrorString() {
		return (new String(getErrorBytes()));
	}

	private void copyStream(InputStream in, OutputStream out) throws java.io.IOException {
		byte[] buf = new byte[BUFSIZE];
		int i = in.read(buf);
		while (i > 0) {
			out.write(buf, 0, i);
			i = in.read(buf);
		}
	}
	
	private InterruptedException interrupted() {
		return (new InterruptedException("Process killed after " + maxRuntimeMillis + " ms."));
	}
	
	public void waitFor() throws IOException, InterruptedException {
		try {
			ByteArrayOutputStream bout = new ByteArrayOutputStream();
			ByteArrayOutputStream berr = new ByteArrayOutputStream();
			copyStream(out, bout);
			copyStream(err, berr);
			outBytes = bout.toByteArray();
			errBytes = berr.toByteArray();
			process.waitFor();
		} catch (IOException e) {
			if (killed) throw (interrupted());
			throw (e);
		} catch (InterruptedException e) {
			throw (e);
		} catch (Throwable t) {
			if (killed) {
				throw (interrupted());
			}
		}
	}
	
	public static void main(String[] args) throws Throwable {
		ProcessResult res = new ProcessResult("sudo locate -u", 3000);
		res.waitFor();
		System.out.println("Output:");
		System.out.write(res.getOutputBytes());
		System.out.println("\n\nErrors:");
		System.out.write(res.getErrorBytes());
		System.out.println("\n\nExit Code: " + res.getProcess().exitValue());
		
	}

	private class ProcessKiller extends TimerTask {
		private ProcessResult processResult = null;
		
		ProcessKiller(ProcessResult processResult) {
			this.processResult = processResult;
		}
		
		public void run() {
			try {
				Process process = processResult.getProcess();

				try {
					int exitValue = process.exitValue();
				} catch (IllegalThreadStateException e) {
					processResult.killed = true;
					process.destroy();
				}
			} catch (Throwable t) {}
		}
}

}