/*
 * Copyright (c) 2005, John Mutchek
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the following 
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials provided 
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote 
 *    products derived from this software without specific prior 
 *    written permission.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR 
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED 
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND 
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Vonage is a reserved mark of Vonage Holdings Corp. which has no
 * affiliation with this open project.  As this source may make 
 * use of services exposed by Vonage to Vonage customers at 
 * http://www.vonage.com, users should be aware of,
 * and heed, the Vonage Terms of Service as described at
 * http://www.vonage.com/features_terms_service.php
 * 
 * Google is a reserved mark of Google which has no affiliation with
 * this project.  As this source may make use of services exposed
 * by Google to the public, users should be aware of, and heed,
 * the Google Terms of Service as described at
 * http://www.google.com/intl/en/terms_of_service.html
 * 
 */

package com.mutchek.vonaje;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.xml.sax.SAXException;

import com.meterware.httpunit.WebConversation;
import com.meterware.httpunit.WebResponse;

/**
 * Represents a voicemail message on the Vonage voicemail system
 * 
 * @author jmutchek
 */

public class Message {

	/**
	 * Phone number of the calling party
	 */
	private PhoneNumber callerNumber = null;
	
	/**
	 * Name of the calling party (if available in the Google Phonebook) 
	 */
	private String callerName = "";
	
	/**
	 * Street address of the calling party (if available in the Google Phonebook) 
	 */
	private String callerAddress = "";
	
	/**
	 * Time the message was left
	 */
	private Date time = null;
	
	/**
	 * Length of the message in seconds
	 */
	private int duration = 0;
	
	/**
	 * URL of the original voicemail file on the Vonage server
	 */
	private String url = "";
	
	/**
	 * Flag indicating whether this message is new
	 */
	private boolean newFlag = false;
	
	/**
	 * Size (in bytes) of the cached message file
	 */
	private int originalSize = 0;
	
	/**
	 * File system location of the message file cache
	 */
	private String cache = "";
	
	/**
	 * Phone vonagePhoneNumber with which this message is associated
	 */
	private VonagePhoneNumber vonagePhoneNumber = null;
	
	/**
	 * Numeric id assigned by vonage to this message (1-based, starting with oldest message)
	 */
	private int listID = 0;
	
	/**
	 * Delete the message from the server
	 */
	public void delete() throws VonageConnectivityException {
		vonagePhoneNumber.deleteMessage(listID);
		removeFromCache();
	}

	/**
	 * Retrieve the date and time the message was left in the 
	 * yyyyMMddhhmma format.  e.g. 100512050100PM is returned for
	 * a message left at 1:00 PM on Dec. 5, 2005.
	 * 
	 * @return Formatted datetime string
	 */
	public String getDateString() {
		SimpleDateFormat df = new SimpleDateFormat("yyyyMMddhhmma");
		return df.format(time);
	}
	
	/**
	 * Associate the message with the Vonage phone number for
	 * which the voicemail was left
	 * 
	 * @param the VonagePhoneNumber associated with this message
	 */
	protected void setVonagePhoneNumber(VonagePhoneNumber number) {
		this.vonagePhoneNumber = number;
	}
	
	/**
	 * Retrieve the Vonage phone number for which the voicemail
	 * was left
	 * 
	 * @return the VonagePhoneNumber associated with this message
	 */
	public VonagePhoneNumber getVonagePhoneNumber() {
		return vonagePhoneNumber;
	}
	
	/**
	 * Check if the message is new
	 * 
	 * @return true if the message has never been heard, false otherwise
	 */
	public boolean isNew() {
		return newFlag;
	}

	/**
	 * Set the vonage message list id
	 * @param id
	 */
	protected void setListID(int id) {
		listID = id;
	}
	
	/**
	 * Retrieve the vonage assigned list id
	 * @return
	 */
	public int getListID() {
		return listID;
	}
	
	/**
	 * Set the new flag on the message
	 * @param newFlag true if the message has never been heard
	 */
	protected void markNew(boolean newFlag) {
		this.newFlag = newFlag;
	}
	
	/**
	 * Retrieve the 11-digit phone number of the calling party
	 * 
	 * @return the caller's phone number
	 */
	public String getCallerNumber() {
		return callerNumber.getNumber();
	}
	
	/**
	 * Retrieve the directory listing name of the calling party
	 * as listed in the Google Phonebook
	 * 
	 * @return the caller's name
	 */
	public String getCallerName() {
		return callerName;
	}
	 
	/**
	 * Retrieve the datetime the message was left
	 * 
	 * @return the datetime the message was left
	 */
	public Date getCallTime() {
		return time;
	}
	
	/**
	 * Set the caller's 11-digit phone number
	 * @param callerNumber the 11-digit phone number of the caller
	 * @throws InvalidPhoneNumberException if anything but an 11-digit numeric string is passed
	 */
	protected void setCallerNumber(String callerNumber) throws InvalidPhoneNumberException{
		this.callerNumber = new PhoneNumber(detag(callerNumber));
	}
	
	/**
	 * Retrieve the caller's address as listed in the Google Phonebook
	 * @return the caller's address
	 */
	public String getCallerAddress() {
		return callerAddress;
	}
	
	/**
	 * Lookup the caller's phone number using the
	 * Google Phonebook service. If the number is listed,
	 * use the name and address
	 */
	protected void applyCallerID() {
		if (callerNumber.isValidNumber()) {
			WebConversation google = new WebConversation();
			try {
				WebResponse result = google.getResponse("http://www.google.com/search?q=" + callerNumber.getNumber().substring(1));
				String html = result.getText();
				Pattern p = Pattern.compile("<font size=-1>(.*?),.*?,(.*?)<a");
				Matcher m = p.matcher(html);
				if (m.find()) {
					setCallerName(m.group(1));
					setCallerAddress(m.group(2).trim());
				}
			} catch (MalformedURLException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (SAXException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();			
			}
		} else {
//			System.err.println("Invalid or unknown number, skipping caller id.");
		}
	}
	
	private void setCallerName(String callerName) {
		this.callerName = detag(callerName);
	}
	
	private void setCallerAddress(String callerAddress) {
		this.callerAddress = detag(callerAddress);
	}
	
	/**
	 * Set the datetime the message was left 
	 * @param callTime the datetime the message was left
	 */
	protected void setCallTime(String callTime) {
		SimpleDateFormat df = new SimpleDateFormat(vonagePhoneNumber.getVonageAccount().getCountryProperties().getProperty("date_format"));
		try {
			time = df.parse(callTime);
		} catch (ParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	/**
	 * Set the duration of the message
	 * @param duration the duration of the message in seconds
	 */
	protected void setDuration(int duration) {
		this.duration = duration;
	}
	
	/**
	 * Set the duration of the message
	 * @param duration the duration of the message expressed in mm:ss
	 */
	protected void setDuration(String length) {
		String parts[] = length.split(":");
		int minutes = Integer.parseInt(parts[0]);
		int seconds = Integer.parseInt(parts[1]);
		duration = (minutes * 60) + seconds;
	}
	
	/**
	 * Retrieve the duration of the message in seconds
	 * @return the duration of the message in seconds
	 */
	public int getDuration() {
		return duration;
	}
	
	/**
	 * Set the URL at which the message can be downloaded
	 * from Vonage
	 * @param URL of the message file
	 */
	protected void setURL(String url) {
		this.url = url;
	}
	
	/**
	 * Retrieve the URL of the message at Vonage
	 * @return the URL of the message file
	 */
	public String getURL() {
		return url;
	}

	/**
	 * Retrieve a string representation of the caller which
	 * contains either the caller name and number or just the
	 * number (if caller id is not available)
	 * @return a string representation of the caller
	 */
	public String getCallerString() {
		String rv = "";
		
		if (callerName.equals("")) {
			rv = callerNumber.format();
		} else {
			rv = callerName + " (" + callerNumber.formatWithoutParens() + ")";
		}
		
		return rv;		
	}
	
	/**
	 * Exactly the same as getCallerString()
	 */
	public String toString() {
		return getCallerString();
	}
	
	/**
	 * Retrieve a message wav file
	 * @param index
	 * @return
	 */
	public File getMessage(ProgressHandler handler) throws OrphanedMessageException {
		WebConversation wwwVonageSession = null;
		
		if (vonagePhoneNumber == null) {
			throw new OrphanedMessageException();
		}
		
		try {
			wwwVonageSession = vonagePhoneNumber.getVonageAccount().getVonageSession();
		} catch (VonageConnectivityException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		File cachedFile = null;
		
		File numberCache = new File(vonagePhoneNumber.getMessageCache() + vonagePhoneNumber.getNumber());
		if (!numberCache.exists()) {
			numberCache.mkdir();			
		}
		
		cachedFile = new File(vonagePhoneNumber.getMessageCache() + vonagePhoneNumber.getNumber() + File.separator + getDateString() + "_" + callerNumber.getNumber() + ".wav");
		if (!cachedFile.exists()) {
			WebResponse messageFileResponse;
			try {
				messageFileResponse = wwwVonageSession.getResponse(getURL());
				InputStream in = messageFileResponse.getInputStream();
				FileOutputStream out = new FileOutputStream(cachedFile.getAbsolutePath());
				int b;
				float estimatedSize = duration * 15000;
				int byteCount = 0;
				while ((b = in.read()) >= 0) {
					byteCount++;
					out.write(b);
					if ((byteCount%16384)==0) {
						handler.setProgress(byteCount/estimatedSize);
					}
				}
				out.flush();
				out.close();				
			} catch (MalformedURLException e1) {
				System.out.println("MalformedURL");
				e1.printStackTrace();
			} catch (IOException e1) {
				System.out.println("IOException");
				e1.printStackTrace();
			} catch (SAXException e1) {
				System.out.println("SAXException");
				e1.printStackTrace();
			}
		}			
		handler.setComplete(true);
		markNew(false);
		return cachedFile;
	}
	
	public boolean isCached() {
		File cachedFile = null;
		cachedFile = new File(vonagePhoneNumber.getMessageCache() + vonagePhoneNumber.getNumber() + File.separator + getDateString() + "_" + callerNumber.getNumber() + ".wav");
		return cachedFile.exists();
	}

	/**
	 * Delete the locally cached file
	 */
	public void removeFromCache() {
		File cachedFile = null;
		String msgFilenameNoExt = getDateString() + "_" + callerNumber.getNumber();
		try {
			File[] cacheFiles = new File(vonagePhoneNumber.getMessageCache() + vonagePhoneNumber.getNumber()).listFiles();
			for (int i = 0; i < cacheFiles.length; ++i) {
				if (cacheFiles[i].getName().startsWith(msgFilenameNoExt)) cacheFiles[i].delete();
			}
		} catch (Exception ignored) {}
	}
	
	private String detag(String s) {
		return (s == null ? s : s.replaceAll("<.*?>",""));
	}
}
