/*
 * Copyright (c) 2004, Martin Lamb
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright 
 * notice, this list of conditions and the following disclaimer in the 
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Martin Lamb nor the name of Martian Software, Inc.
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.martiansoftware.snip;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.Iterator;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.FileSet;

/**
 * An <a href="http://ant.apache.org">Ant</a> task designed to help with the
 * single-sourcing of program documentation.  This task extracts snippets
 * of text from files, placing them into properties in the Ant project.
 * These properties can then be used by any other Ant task, and are
 * particularly useful when referenced by <code>&lt;filter&gt;</code>s
 * within the <code>&lt;copy&gt;</code> task.
 * 
 * <p>The <code>&lt;snip&gt;</code> task takes one or more nested
 * <code>&lt;fileset&gt;</code>s that indicate the
 * files to process.  Any text between lines containing "@@snip:<i>myProperty</i>@@"
 * and "@@endSnip@@" will be stored in a property called snip.<i>myProperty</i>.</p>
 * 
 * <p>Example Usage:<pre>
 * <code>
 * &lt;taskdef
 *     name="snip"
 *     classname="com.martiansoftware.snip.Snip" 
 *     classpath="@@JARNAME@@"/&gt;
 * 		
 * &lt;snip&gt;
 *     &lt;fileset dir="src" includes="*&#042;/*.txt" /&gt;
 * &lt;/snip&gt;
 * </code></pre>
 * </p>
 * 
 * <p>Multiple snippets may be defined within a single file.  Snippets may
 * not be nested.</p>
 * 
 * @author <a href="http://www.martiansoftware.com/contact.html">Marty Lamb</a>
 */
public class Snip extends Task {

	/**
	 * Text indicating the beginning of a snippet declaration
	 */
	private static final String SNIP_START_OPEN = "@@snip:";
	
	/**
	 * Text closing a snippet declaration
	 */
	private static final String SNIP_START_CLOSE = "@@";
	
	/**
	 * Text closing a snippet
	 */
	private static final String SNIP_END = "@@endSnip@@";
	
	/**
	 * Prefix prepended to all property names set by this task
	 */
	private static final String PROPERTY_PREFIX = "snip.";
	
	/**
	 * Stores all the FileSets provided by Ant
	 */
	private ArrayList filesets = new ArrayList();
	
	/**
	 * Just use a single StringBuffer for all processing.
	 */
	StringBuffer buf = new StringBuffer();
	
	/**
	 * True if the property should be "xml-safe" - that is,
	 * should have any greater-than or less-than signs xml-encoded
	 */
	private boolean xmlSafe = false;
	
	/**
	 * Adds a FileSet to this Task
	 * @param fs the FileSet to add
	 */
	public void addFileSet(FileSet fs) {
		filesets.add(fs);
	}

	/**
	 * Returns true iff the specified String includes the SNIP_END text
	 * @param s the String to test
	 * @return true iff the specified String includes the SNIP_END text
	 */
	private boolean endSnip(String s) {
		return (s.indexOf(SNIP_END) >= 0);
	}
	
	/**
	 * Sets whether properties set by this task should be
	 * "xml-safe" - that is, should have any greater-than or less-than signs
	 * xml-encoded
	 * @param xmlSafe
	 */
	public void setXmlsafe(boolean xmlSafe) {
		this.xmlSafe = xmlSafe;
	}
	
	/**
	 * If the specified String contains a snippet declaration, returns the
	 * name of the snippet.  Otherwise, returns null.
	 * @param s the String to test
	 * @return the name of the snippet that begins on the next line, or null
	 * if the specified String does not contain a snippet declaration.
	 */
	private String startSnip(String s) {
		String result = null;
		int start = 0;
		if ((start = s.indexOf(SNIP_START_OPEN)) >= 0) {
			start += SNIP_START_OPEN.length();
			int end = s.indexOf(SNIP_START_CLOSE, start);
			if (end >= 0) {
				result = s.substring(start, end).trim();
				if (result.length() == 0) {
					result = null;
				}
			}
		}
		return (result);
	}
	
	/**
	 * Processes a single File, reading any snippets it contains and storing
	 * them in project properties
	 * @param file the File to test
	 * @throws BuildException if any I/O errors occur while trying to read
	 * the file
	 */
	private void snipFile(File file) throws BuildException {
		boolean snipping = false;
		String propertyName = null;
		buf.setLength(0);
		
		try {
			LineNumberReader in = new LineNumberReader(new InputStreamReader(new BufferedInputStream(new FileInputStream(file))));
			String s = in.readLine();
			while (s != null) {
				if (snipping) {
					if (endSnip(s)) {
						snipping = false;
						getProject().setProperty(PROPERTY_PREFIX + propertyName, 
								xmlSafe
									? buf.toString().replaceAll("<", "&lt;").replaceAll(">","&gt;")
									: buf.toString());
						propertyName = null;
						buf.setLength(0);
					} else {
						buf.append("\n");
						buf.append(s);
					}
				} else {
					if ((propertyName = startSnip(s)) != null) {
						snipping = true;
					}
				}
				s = in.readLine();
			}
		} catch (Throwable t) {
			throw (new BuildException("Unable to read " + file.getAbsolutePath() + ": " + t.getMessage(), t));
		}
	}

	/**
	 * Reads all snippets from the previously specified FileSets, storing
	 * their contents in project properties.
	 * @throws BuildException if any I/O errors occur while reading files.
	 */
	public void execute() throws BuildException {
		for (Iterator i = filesets.iterator(); i.hasNext();) {
			FileSet fs = (FileSet) i.next();
			File dir = fs.getDir(getProject());
			DirectoryScanner ds = fs.getDirectoryScanner(getProject());
			String[] srcFiles = ds.getIncludedFiles();
			for (int idx = 0; idx < srcFiles.length; ++idx) {
				snipFile(new File(dir, srcFiles[idx]));
			}
		}
	}
	
}
