/*
 * Copyright (c) 2004, Martin Lamb
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright 
 * notice, this list of conditions and the following disclaimer in the 
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Martin Lamb nor the name of Martian Software, Inc.
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.martiansoftware.rundoc;

import java.io.Writer;
import java.io.File;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Iterator;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.Environment;
/**
 * An <a href="http://ant.apache.org">Ant</a> task designed to help with the
 * single-sourcing of program documentation.  This task replaces special
 * commands embedded within text files with their output in a specified format.
 * Currently, only docbook format is supported.
 * 
 * <p>For example, if a text file contains the text
 * "<code>@@rundoc:ls -l@@</code>", and the task is called with the following
 * code:
 * <code><pre>
 * &lt;taskdef
 *     name="rundoc"
 *     classname="com.martiansoftware.rundoc.RunDoc" 
 *     classpath="@@JARNAME@@"/&gt;
 * 
 * &lt;target name="testrundoc"&gt;
 *     &lt;rundoc prompt="[mlamb@morbo]$" format="docbook"&gt;
 *       &lt;fileset dir="." includes="test/*.txt"/&gt;
 *     &lt;/rundoc&gt;
 * &lt;/target&gt;
 * </pre></code> 
 * <p>it might be replaced with something like:</p>
 * <code><pre>
 * &lt;prompt&gt;[mlamb@morbo]$&lt;/prompt&gt;&lt;command&gt;ls -l&lt;/command&gt;
 * &lt;computeroutput&gt;total 40
 * drwxrwxr-x    3 mlamb    mlamb        4096 Feb 15 18:45 build
 * -rw-rw-r--    1 mlamb    mlamb        2365 Feb 15 19:22 build.xml
 * drwxrwxr-x    2 mlamb    mlamb        4096 Feb 15 18:51 CVS
 * drwxrwxr-x    2 mlamb    mlamb        4096 Feb 15 18:49 dist
 * drwxrwxr-x    4 mlamb    mlamb        4096 Feb 15 18:49 javadoc
 * drwxrwxr-x    3 mlamb    mlamb        4096 Feb 15 18:51 lib
 * -rw-rw-r--    1 mlamb    mlamb        1481 Feb  1 19:09 LICENSE.txt
 * -rw-rw-r--    1 mlamb    mlamb         336 Feb  1 20:02 README.txt
 * drwxrwxr-x    4 mlamb    mlamb        4096 Feb 15 18:51 src
 * drwxrwxr-x    2 mlamb    mlamb        4096 Feb 15 19:20 test
 * &lt;/computeroutput&gt;</pre></code>
 * 
 * <p>Replacements are made in whatever files are included in the nested
 * filesets; odds are you'll want to <code>&lt;copy&gt;</code> your files
 * before running rundoc on the copies.<p>
 * 
 * <p>Rundoc supports nested &lt;env&gt; elements to pass environment variables
 * to the executed process(es).  See the documentation for
 * <a href="http://ant.apache.org/manual/CoreTasks/exec.html">&lt;exec&gt;</a>
 * for details on its use.  Here's a simple example that's useful if rundoc
 * will be running part of the current java project:</p>
 * 
 * <code><pre>
 * &lt;target name="rundocs"&gt;
 *     &lt;!-- this example assumes that a temporary copy of your manual.xml
 *          docbook file has already been created in ${build} --&gt;
 *     &lt;rundoc prompt="[mlamb@hypno-toad]$" format="docbook"&gt;
 *       &lt;fileset file="${build}/manual.xml"/&gt;
 *       &lt;env key="CLASSPATH" value="${build}/&gt;
 *     &lt;/rundoc&gt;
 * &lt;/target&gt;
 * </pre></code> 

 * 
 * <p>A typical usage scenario would be to put the 
 * <code>@@rundoc:<i>command</i>@@</code> directly in the docbook source for
 * your documentation between <code>&lt;screen&gt;</code> tags.  The build
 * process would then create a temporary copy of the docbook source, run
 * &lt;rundoc&gt; against it, and finally run the modified docbook file through
 * a formatter.</p>
 * 
 * <p>This task goes hand-in-hand with 
 * <a href="http://www.martiansoftware.com/lab/index.html#snip">
 * <code>&lt;snip&gt;</code></a>.</p>
 * 
 * <p>Multiple rundoc commands may be defined within a single file.</p>
 * 
 * <p><b>Possible Enhancements:</b><br/>
 * I currently have no plans to implement these, but they would probably be
 * useful to someone.  Code contributions are welcome.
 * <ul>
 * <li>Provide a means to specify the directory in which the command should be run.</li>
 * <li>Support additional output formats (man? others?)
 * </ul>
 * </p>
 * 
 * @author <a href="http://www.martiansoftware.com/contact.html">Marty Lamb</a>
 */
public class RunDoc extends Task implements MacroProcessor {

	/**
	 * Format constant indicating that output should be written
	 * in docbook format.
	 */
	public static final String FORMAT_DOCBOOK = "docbook";
	
	/**
	 * Text indicating the beginning of a runsnippet declaration
	 */
	private static final String RUNSNIP_START = "@@rundoc:";
	
	/**
	 * Text closing a snippet declaration
	 */
	private static final String RUNSNIP_END = "@@";

	/**
	 * Size of buffer for reading file chunks
	 */
	private static final int BUFSIZE = 2048;
	
	/**
	 * The prompt to display in the output
	 */
	private String prompt = null;
	
	/**
	 * The output format to use
	 */
	private String format = FORMAT_DOCBOOK;
	
	/**
	 * Stores all the FileSets provided by Ant
	 */
	private ArrayList filesets = new ArrayList();
	
	/**
	 * Just use a single StringBuffer for all processing.
	 */
	StringBuffer buf = new StringBuffer();
	
	/**
	 * The environment to pass to the executed process(es)
	 */
	private Environment env = new Environment();
	
	/**
	 * Adds a FileSet to this Task
	 * @param fs the FileSet to add
	 */
	public void addFileSet(FileSet fs) {
		filesets.add(fs);
	}

	/**
	 * Makes the specified string safe for insertion into an xml file
	 * @param s the String to make safe
	 * @return a safe version
	 */
	private String xmlSafe(String s) {
		return (s.replaceAll("<", "&lt;").replaceAll(">", "&gt;"));
	}
	
	/**
	 * Add an environment variable.
	 *
	 * @param envvar new environment variable
	 */
	public void addEnv(Environment.Variable envvar) {
		env.addVariable(envvar);
	}
	
	/**
	 * Sets the prompt to include in output
	 * @param prompt the prompt to include in output
	 */
	public void setPrompt(String prompt) {
		this.prompt = prompt;
	}
	
	/**
	 * Formats the program output in docbook format.
	 * @param command the command executed
	 * @param output the command's output
	 * @return the program output in docbook format (minus
	 * enclosing &lt;screen&gt; tags)
	 */
	public String formatDocbook(String command, String output) {
		StringBuffer result = new StringBuffer();
		result.append("<prompt>");
		result.append(xmlSafe(prompt));
		result.append("</prompt><command>");
		result.append(xmlSafe(command));
		result.append("</command>\n<computeroutput>");
		result.append(xmlSafe(output));
		result.append("</computeroutput>");
		return (result.toString());
	}

	/**
	 * Formats the program output as specified by the user (currently only
	 * docbook format is supported)
	 * @param command the command executed
	 * @param output the command's output
	 * @return the formatted output
	 */
	public String formatResult(String command, String output) {
		// for now, the only possible format is docbook.  as
		// other formats are added, it will be necessary to
		// add checks here
		if (format.equalsIgnoreCase(FORMAT_DOCBOOK)) {
			return (formatDocbook(command, output));
		}
		throw (new BuildException("Unsupported format: " + format));
	}
	
	/**
	 * Sets the output format
	 * @param format the output format (currently only "docbook" is
	 * supported)
	 */
	public void setFormat(String format) {
		this.format = format;
	}
	
	/**
	 * Processes the specified macro, returning the formatted result.
	 * @param macro the macro to run
	 * @return the formatted output of the specified macro
	 */
	public String processMacro(String macro) {
		System.out.println("Executing [" + macro + "]...");
		StringWriter result = new StringWriter();
		try {
			Process p = Runtime.getRuntime().exec(macro, env.getVariables());
			Reader in = new java.io.BufferedReader(new java.io.InputStreamReader(p.getInputStream()));
			Reader errIn = new java.io.BufferedReader(new java.io.InputStreamReader(p.getErrorStream()));
			
			char[] buf = new char[BUFSIZE];
			
			int charsRead = 0;
			int errCharsRead = 0;
			while ((charsRead >= 0) || (errCharsRead >= 0)) {
				if ((charsRead = in.read(buf)) >= 0) {
//					System.out.println("Read " + charsRead + " chars");
					result.write(buf, 0, charsRead);
				}
				if ((errCharsRead = errIn.read(buf)) >= 0) {
//					System.err.println("Read " + errCharsRead + " error chars");
					result.write(buf, 0, errCharsRead);
				}
			}
			
			p.waitFor();
			in.close();
		} catch (Throwable t) {
			t.printStackTrace();
			throw (new BuildException("Error running [" + macro + "]", t));
		}
		return (formatResult(macro, result.toString()));
	}
	
	/**
	 * Processes a single File, reading any snippets it contains and storing
	 * them in project properties
	 * @param file the File to test
	 * @throws BuildException if any I/O errors occur while trying to read
	 * the file
	 */
	private void runSnipFile(File file) throws BuildException {
		if (prompt == null) {
			throw (new BuildException("No prompt set."));
		}
		try {
			InputStream fin = new java.io.BufferedInputStream(new java.io.FileInputStream(file));
			MacroInputStream min = new MacroInputStream(fin, this);
			min.setMacroDelimiters(RUNSNIP_START, RUNSNIP_END);
			Reader in = new java.io.InputStreamReader(min);
			StringWriter result = new StringWriter();
			char[] buf = new char[BUFSIZE];
			int charsRead = 0;
			while ((charsRead = in.read(buf)) >= 0) {
				result.write(buf, 0, charsRead);
			};
			in.close();
			result.close();
			
			Writer out = new java.io.OutputStreamWriter(new java.io.BufferedOutputStream(new java.io.FileOutputStream(file)));
			out.write(result.toString());
			out.close();
		} catch (Throwable t) {
			throw (new BuildException("Unable to read " + file.getAbsolutePath() + ": " + t.getMessage(), t));
		}
	}

	/**
	 * Reads all snippets from the previously specified FileSets, storing
	 * their contents in project properties.
	 * @throws BuildException if any I/O errors occur while reading files.
	 */
	public void execute() throws BuildException {
		for (Iterator i = filesets.iterator(); i.hasNext();) {
			FileSet fs = (FileSet) i.next();
			File dir = fs.getDir(getProject());
			DirectoryScanner ds = fs.getDirectoryScanner(getProject());
			String[] srcFiles = ds.getIncludedFiles();
			for (int idx = 0; idx < srcFiles.length; ++idx) {
				runSnipFile(new File(dir, srcFiles[idx]));
			}
		}
	}
	
}
