/*
 * Copyright (c) 2004, Martin Lamb
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright 
 * notice, this list of conditions and the following disclaimer in the 
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Martin Lamb nor the name of Martian Software, Inc.
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, 
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY 
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.martiansoftware.rundoc;

import java.io.IOException;
import java.io.InputStream;
import java.io.FilterInputStream;

/**
 * TODO: write docs.
 * @author <a href="http://www.martiansoftware.com/contact.html">Marty Lamb</a>
 */
class MacroInputStream extends FilterInputStream {

	/**
	 * The default maximum lookahead when reading macros.
	 */
	public static final int DEFAULT_MAXMACROLENGTH = 1024;
	
	/**
	 * The default indicator of the beginning of a macro.
	 */
	public static final String DEFAULT_MACROSTART = "${";
	
	/**
	 * The default indicator of the end of a macro.
	 */
	public static final String DEFAULT_MACROEND = "}";
	
	/**
	 * The marker of the beginning of a macro.
	 */
	private String macroStart = DEFAULT_MACROSTART;
	
	/**
	 * The length of the macro start delimiter (to avoid recomputing).
	 */
	private int macroStartLength = DEFAULT_MACROSTART.length();
	
	/**
	 * The marker of the end of a macro.
	 */
	private String macroEnd = DEFAULT_MACROEND;

	/**
	 * The length of the macro end delimiter (to avoid recomputing).
	 */
	private int macroEndLength = DEFAULT_MACROEND.length();
	
	/**
	 * The maximum lookahead when reading macros.
	 */
	private int maxMacroLength = DEFAULT_MAXMACROLENGTH;
	
	/**
	 * The size of the buffer to use.
	 */
	private int bufSize = DEFAULT_MAXMACROLENGTH 
							+ macroStartLength
							+ macroEndLength;
	
	/**
	 * The maximum size to which the buffer can grow.  Uncontrolled
	 * growth could be a symptom of circular symbol definitions.
	 * For example ${MYPROPERTY}="ABC${MYPROPERTY}".  This is obviously
	 * problematic, so a MacroException will be thrown if the buffer
	 * ever expands to this size.
	 * 
	 * The only reason the buffer should ever grow beyond bufSize is
	 * via macro insertion.
	 */
	private int maxBufSize = 32767;
	
	/**
	 * The maximum left recursive depth permitted by the macro
	 * processor.  "Left recursion" in this context refers to the
	 * replacement of macros with strings that begin with macros at
	 * the first character.
	 * 
	 * For example, if the symbol "${A}" evaluates to "${B}"
	 * and "${B}" evaluates to "Hello", and the underlying reader begins with
	 * the String "${A}, World!", the buffer will go through the following
	 * states:
	 * 
	 * 		${A}, World!
	 * 		${B}, World!
	 * 		Hello, World!
	 * 
	 * before the first character is returned.  Note that the left recursion
	 * depth only increments when a macro evaluates to a string containing
	 * a macro beginning at the first character; if ${A} contained "hello ${B}",
	 * that would not be a left recursion issue, and uncontrolled growth would
	 * be handled by maxBufSize, above.
	 */
	private int maxLeftRecursionDepth = 32;
	
	/**
	 * The processor for any macros encountered.
	 */
	private MacroProcessor processor = null;
	
	/**
	 * The buffer we'll use to read ahead.
	 */
	private StringBuffer buf = null;
	
	/**
	 * If true, the underlying Reader is empty.
	 */
	private boolean eof = false;
	
	/**
	 * Creates a new MacroInputStream wrapping the specified InputStream and using
	 * the specified MacroProcessor.
	 * 
	 * @param in the InputStream to wrap.
	 * @param processor the MacroProcessor to use.
	 */	
	public MacroInputStream(InputStream in, MacroProcessor processor) {
		super(in);
		init(processor, DEFAULT_MAXMACROLENGTH);		
	}

	/**
	 * Creates a new MacroInputStream wrapping the specified InputStream and using
	 * the specified MacroProcessor and maximum macro lookahead.
	 * 
	 * @param in the InputStream to wrap.
	 * @param processor the MacroProcessor to use.
	 * @param maxMacroLength the maximum macro lookahead to use.
	 */	
	public MacroInputStream(InputStream in, 
						MacroProcessor processor, 
						int maxMacroLength) {
		super(in);
		init(processor, maxMacroLength);
	}

	/**
	 * Sets the processor and macro lookahead.
	 * 
	 * @param processor the MacroProcessor to use
	 * @param maxMacroLength the max macro lookahead to use.
	 */
	private void init(MacroProcessor processor, int maxMacroLength) {
		this.processor = processor;
		setMaxMacroLength(maxMacroLength);
		buf = new StringBuffer();
		eof = false;
	}
	
	/**
	 * Recomputes the size of the read buffer.
	 */
	private void recomputeBufSize() {
		bufSize = maxMacroLength + macroStartLength + macroEndLength;
	}
	
	/**
	 * Sets the max macro lookahead to use.
	 * 
	 * @param maxMacroLength the max macro lookahead to use.
	 */
	public void setMaxMacroLength(int maxMacroLength) {
		this.maxMacroLength = maxMacroLength;
		recomputeBufSize();
	}
	
	/**
	 * Returns the max macro lookahead.
	 * 
	 * @return the max macro lookahead.
	 */
	public int getMaxMacroLength() {
		return (maxMacroLength);
	}

	/**
	 * Sets the macro start indicator.
	 * 
	 * @param macroStart the macro start indicator.
	 */
	public void setMacroStart(String macroStart) {
		if ((macroStart == null) || (macroStart.length() == 0)) {
			throw 
				(new IllegalArgumentException("macroStart may not be empty."));
		}
		this.macroStart = macroStart;
		this.macroStartLength = macroStart.length();
		recomputeBufSize();
	}
	
	/**
	 * Sets the macro end indicator.
	 * 
	 * @param macroEnd the macro end indicator.
	 */
	public void setMacroEnd(String macroEnd) {
		if ((macroEnd == null) || (macroEnd.length() == 0)) {
			throw 
				(new IllegalArgumentException("macroEnd may not be empty."));
		}
		this.macroEnd = macroEnd;
		this.macroEndLength = macroEnd.length();
		recomputeBufSize();
	}
	
	/**
	 * Returns the macro start delimiter.
	 * 
	 * @return the macro start delimiter.
	 */
	public String getMacroStart() {
		return (macroStart);
	}
	
	/**
	 * Returns the macro end delimiter.
	 * 
	 * @return the macro end delimiter.
	 */
	public String getMacroEnd() {
		return (macroEnd);
	}

	/**
	 * Sets the macro start and end delimiters.
	 * 
	 * @param macroStart the macro start delimiter.
	 * @param macroEnd the macro end delimiter.
	 */	
	public void setMacroDelimiters(String macroStart, String macroEnd) {
		setMacroStart(macroStart);
		setMacroEnd(macroEnd);
	}

	/**
	 * Fills the buffer if possible.
	 * 
	 * @param force if true, forces the filling of the buffer regardless
	 * of its current size.
	 * @throws IOException if an IO error occurs in the underlying Reader.
	 */
	private void fillBuffer(boolean force) throws IOException {
		if (!eof) {
			
			// only fill the buffer if it'll be shorter than the
			// macro start delimiter, or if we suspect the beginning
			// of a macro
			if (force || (buf.length() <= macroStartLength)) {
				int charsToRead = bufSize - buf.length();
				
				int charsRead = 0;
				int thisChar = 0;
				while ((!eof) && (charsRead < charsToRead)) {
					thisChar = in.read();		
					if (thisChar == -1) {
						eof = true;
					} else {
						buf.append((char) thisChar);
						++charsRead;
					}
				}
			}
		}
	}
		
	/**
	 * Returns the next character from the underlying InputStream, performing macro
	 * evaluations along the way.
	 * 
	 * @return the next character from the underlying InputStream(with macros
	 * evaluated), or -1 if the end of the stream has been reached.
	 * 
	 * @throws IOException if an IO error occurs.
	 * 
	 */
	public int read() throws IOException {
		fillBuffer(false);
		if (buf.length() == 0) {
			return (-1);
		} else if (buf.length() > macroStartLength) {
			
			// track how many macros we have replaced at the first position
			// in the buffer.  too many and there's most likely a problem
			// with the macro definitions.
			int curLeftRecursionDepth = 0;
			
			// this allows us a way out of the while loop if a macroprocessor
			// chooses not to process a macro.
			boolean done = false;
			while (buf.substring(0, macroStartLength).equals(macroStart)
					&& !done) {

				++curLeftRecursionDepth;
				if (curLeftRecursionDepth >= maxLeftRecursionDepth) {
					throw (new MacroException(
							"Maximum left recursion depth of "
							+ maxLeftRecursionDepth
							+ " has been reached.  You probably have circular"
							+ " macro definitions."));
				}

				// force the buffer to be full so we can find the end of the
				// current macro
				fillBuffer(true);

				int endPos = buf.toString().indexOf(macroEnd, macroStartLength);
				if (endPos > -1) {
					String macro = buf.substring(macroStartLength, endPos);
					if (processor != null) {
						// the macroprocessor returns a String containing
						// the evaluated macro whose contents then directly 
						// replace the original macro in the buffer.
						String macroResult = processor.processMacro(macro);

						// if the macroprocessor didn't process the macro, just
						// leave the macro in the buffer.
						if (macroResult == null) {
							done = true;
						} else {
							buf.delete(0, endPos + macroEndLength);
							buf.insert(0, macroResult);
							if (buf.length() >= maxBufSize) {
								throw (new MacroException(
									"Maximum buffer size of "
									+ maxBufSize
									+ " has been reached.  You probably have"
									+ " circular macro definitions."));
							}
						}	
					}
				}
			}			
		}
		int result = buf.charAt(0);
		buf.delete(0, 1);
		return (result);
	}

	/**
	 * Reads into the specified byte array, filling it if possible.
	 * @param b the array into which to read
	 * @return the number of bytes read, or -1 if nothing was read and the end of
	 * the stream has been reached.
	 * @throws IOException if an I/O error occurs
	 */
	public int read(byte[] b) throws IOException {
			return (read(b, 0, b.length));
	}
	
	/**
	 * Attempts to read <code>length</code> bytes into the specified byte array,
	 * beginning at index <code>offset</code>.
	 * @param b the array into which to read
	 * @param offset the beginning index at which the array should be filled
	 * @param length the maximum number of bytes to read
	 * @return the number of bytes read, or -1 if nothing was read and the end of
	 * the stream has been reached.
	 * @throws IOException if an I/O error occurs
	 */
	public int read(byte[] b, int offset, int length) throws IOException {
		int result = 0;
		if (!eof) {
			for (int curIndex = offset; curIndex + result < length; ++curIndex) {
				int c = read();
				if (c == -1) {
					break;
				}
				b[curIndex] = (byte) c;
				++result;
			}
		}
		if ((result == 0) && (length != 0)) {
			result = -1;
		}
		return (result);
	}
	
	/**
	 * Returns the maxBufSize.
	 * @return int
	 */
	public int getMaxBufSize() {
		return maxBufSize;
	}

	/**
	 * Returns the maxLeftRecursionDepth.
	 * @return int
	 */
	public int getMaxLeftRecursionDepth() {
		return maxLeftRecursionDepth;
	}

	/**
	 * Sets the maxBufSize.
	 * @param maxBufSize The maxBufSize to set
	 */
	public void setMaxBufSize(int maxBufSize) {
		this.maxBufSize = maxBufSize;
	}

	/**
	 * Sets the maxLeftRecursionDepth.
	 * @param maxLeftRecursionDepth The maxLeftRecursionDepth to set
	 */
	public void setMaxLeftRecursionDepth(int maxLeftRecursionDepth) {
		this.maxLeftRecursionDepth = maxLeftRecursionDepth;
	}

}
